% script to plot an animation of the simulated BM#1 data
% users should be able to see the structure of the model data, and modify
% to suit their debris transport models.

clear all
clf

% numerical data files
eta_fname='zeta.nc';   % contains free surface elevation data
velo_fname='velo.nc';   % contains velocity data
blvs_fname='blvs.nc';   % contains dry/wet and breaking eddy viscosity data

time_numerical=ncread(eta_fname,'time');  % time, length [nt]
x_numerical=ncread(eta_fname,'x');  % x-coord, length [nx]
depth=ncread(eta_fname,'depth');  % depth profile, length [nx]
zeta=squeeze(ncread(eta_fname,'zeta'));  % free surface elevation [nx by nt], squeeze to get rid of y-dimension index
u_vel=squeeze(ncread(velo_fname,'u_velo'));  % cross-shore velocity [nx by nt], squeeze to get rid of y-dimension index
wet_dry=squeeze(ncread(blvs_fname,'boundary_boolean'));  % wet/dry surface, =0 when wet, =99 when dry [nx by nt], squeeze to get rid of y-dimension index
nu_breaking=squeeze(ncread(blvs_fname,'eddy_viscosity'));  % breaking eddy viscosity, m^2/s [nx by nt], squeeze to get rid of y-dimension index

% numerical shift values - numerical domain was not same as physical domain
x_shift=-10.;  % added 10-m to offshore boundary to create wave
t_shift=20-1.4;  % cut off first 20 ish seconds since nothing happened;

time=time_numerical+t_shift;
x=x_numerical+x_shift;

% turn dry points into NaN for plotting
iNaN=find(wet_dry==99);
zeta(iNaN)=NaN;
u_vel(iNaN)=NaN;
nu_breaking(iNaN)=NaN;

% animate data
nt=length(time);
for n=1:10:nt  % plotting every 10th time step
    clf
    subplot(2,1,1)
    plot(x,-depth,'r','LineWidth',3)
    hold on
    plot(x,zeta(:,n),'b','Linewidth',3)

    axis([0 44 -1 0.5])
    xlabel('Distance from Wavemaker (m)')
    ylabel('Elevation wrt SWL (m)')
    title(['Water Elevation at Time (s) ' num2str(time(n))])
    legend('Bottom Profile','Water Surface Profile')

    subplot(2,1,2)
    yyaxis left
    plot(x,u_vel(:,n),'b','LineWidth',3)
    hold on
    axis([0 44 -2 3])
    ylabel('Cross-shore Velocity (m/s)')
    
    yyaxis right
    plot(x,nu_breaking(:,n),'r','Linewidth',3)
    axis([0 44 0 1])
    xlabel('Distance from Wavemaker (m)')
    ylabel('Breaking Eddy Viscosity (m^2/s)')
    title(['Cross-shore Velocity and Eddy Viscosity at Time (s) ' num2str(time(n))])
    legend('Cross-shore Velocity Profile','Eddy Viscosity Profile')

    pause(.01)
end